<?php
/* --------------------------------------------------------------
 LegacyConfigurationController.php 2019-12-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2019 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Configuration\Controller;

use Gambio\Admin\Application\Http\AdminController;
use Gambio\Core\Application\ValueObjects\Url;
use Gambio\Core\Configuration\Models\Read\ConfigurationGroupId;
use Gambio\Core\Configuration\Repositories\ConfigurationGroupRepository;
use Slim\Http\Response;
use Slim\Http\ServerRequest;

/**
 * Class LegacyConfigurationController
 * @package Gambio\Core\Configuration
 *
 * @codeCoverageIgnore
 */
class LegacyConfigurationController extends AdminController
{
    private const MAPPING_CONFIG_EXCLUDE = [
        // my shop
        1  => [
            'configuration/DEFAULT_CUSTOMERS_STATUS_ID_ADMIN',
            'configuration/DEFAULT_CUSTOMERS_STATUS_ID_GUEST',
            'configuration/CURRENT_TEMPLATE'
        ],
        // image options
        4  => [
            'configuration/IMAGE_MANIPULATOR'
        ],
        
        // shipping options
        7  => [
            'configuration/SHIPPING_INFOS'
        ],
        
        // session options
        15 => [
            'configuration/SESSION_FORCE_COOKIE_USE'
        ]
    ];
    
    private const MAPPING_CONTENT_NAVIGATION = [
        [
            'gIds'  => [4],
            'items' => [
                [
                    'name' => ['BOX_CONFIGURATION_4', 'admin_general'],
                    'url'  => './configuration.php?gID=4',
                ],
                [
                    'name' => ['image_processing_title', 'image_processing'],
                    'url'  => './admin.php?do=ImageProcessing',
                ]
            ]
        ],
        [
            'gIds'  => [5],
            'items' => [
                [
                    'name' => ['BOX_CUSTOMERS', 'admin_general'],
                    'url'  => './customers.php',
                ],
                [
                    'name' => ['BOX_CONFIGURATION_5', 'admin_general'],
                    'url'  => './configuration.php?gID=5',
                ]
            ]
        ],
        [
            'gIds'  => [7],
            'items' => [
                [
                    'name' => ['BOX_SHIPPING', 'admin_general'],
                    'url'  => './modules.php?set=shipping',
                ],
                [
                    'name' => ['BOX_CONFIGURATION_7', 'admin_general'],
                    'url'  => './configuration.php?gID=7',
                ]
            ]
        ],
        [
            'gIds'  => [10, 14, 15],
            'items' => [
                [
                    'name' => ['BOX_CONFIGURATION_753', 'admin_general'],
                    'url'  => './admin.php?do=ShopKey',
                ],
                [
                    'name' => ['BOX_GM_SECURITY', 'gm_general'],
                    'url'  => './gm_security.php',
                ],
                [
                    'name' => ['BOX_CONFIGURATION_15', 'admin_general'],
                    'url'  => './configuration.php?gID=15',
                ],
                [
                    'name' => ['BOX_CONFIGURATION_14', 'admin_general'],
                    'url'  => './configuration.php?gID=14',
                ],
                [
                    'name' => ['BOX_CONFIGURATION_10', 'admin_general'],
                    'url'  => './configuration.php?gID=10',
                ]
            ]
        ],
        [
            'gIds'  => [12],
            'items' => [
                [
                    'name' => ['emails', 'emails'],
                    'url'  => './admin.php?do=Emails',
                ],
                [
                    'name' => ['BOX_CONFIGURATION_12', 'admin_general'],
                    'url'  => './configuration.php?gID=12',
                ],
                [
                    'name' => ['BOX_GM_EMAILS', 'gm_general'],
                    'url'  => './gm_emails.php',
                ]
            ]
        ],
        [
            'gIds'  => [18],
            'items' => [
                [
                    'name' => ['BOX_ZONES', 'admin_general'],
                    'url'  => './zones.php',
                ],
                [
                    'name' => ['BOX_COUNTRIES', 'admin_general'],
                    'url'  => './countries.php',
                ],
                [
                    'name' => ['BOX_LANGUAGES', 'admin_general'],
                    'url'  => './languages.php',
                ],
                [
                    'name' => ['BOX_TAX_CLASSES', 'admin_general'],
                    'url'  => './tax_classes.php',
                ],
                [
                    'name' => ['BOX_TAX_RATES', 'admin_general'],
                    'url'  => './tax_rates.php',
                ],
                [
                    'name' => ['BOX_CONFIGURATION_18', 'admin_general'],
                    'url'  => './configuration.php?gID=18',
                ],
                [
                    'name' => ['BOX_GEO_ZONES', 'admin_general'],
                    'url'  => './geo_zones.php',
                ],
                [
                    'name' => ['BOX_CURRENCIES', 'admin_general'],
                    'url'  => './currencies.php',
                ],
            ]
        ],
    ];
    
    /**
     * @var ConfigurationGroupRepository
     */
    private $configurationRepository;
    
    /**
     * @var Url
     */
    private $url;
    
    
    /**
     * LegacyConfigurationController constructor.
     *
     * @param ConfigurationGroupRepository $configurationRepository
     * @param Url                          $url
     */
    public function __construct(ConfigurationGroupRepository $configurationRepository, Url $url)
    {
        $this->configurationRepository = $configurationRepository;
        $this->url                     = $url;
    }
    
    
    /**
     * Configuration page by (legacy) group id.
     *
     * @param ServerRequest $request
     * @param Response      $response
     * @param array         $args
     *
     * @return Response
     */
    public function displayConfigurations(ServerRequest $request, Response $response, array $args): Response
    {
        $groupId        = (int)$args['groupId'] ?? 1;
        $group          = ConfigurationGroupId::create($groupId);
        $configExcludes = self::MAPPING_CONFIG_EXCLUDE[$groupId] ?? [];
        
        $configurations = $this->configurationRepository->getByGroup($group, $configExcludes);
        $this->assign('configurations', $configurations->jsonSerialize());
        $this->assign('groupId', $groupId);
        $this->assign('saveBtn', $this->translate('BUTTON_SAVE', 'admin_buttons'));
        
        $pageTitle = $this->translate("BOX_CONFIGURATION_{$groupId}", 'admin_general');
        
        foreach (self::MAPPING_CONTENT_NAVIGATION as $itemInfo) {
            if (in_array($groupId, $itemInfo['gIds'])) {
                foreach ($itemInfo['items'] as $item) {
                    $name    = $this->translate(...$item['name']);
                    $url     = $item['url'];
                    $current = $url === "./configuration.php?gID=$groupId";
                    $url     = $this->url->admin() . ltrim($url, '.');
                    
                    $this->assignContentNavItem($name, $url, $current);
                }
            }
        }
        
        $templatePath = __DIR__ . '/ui/configurations.html';
        $template     = $this->render($pageTitle, $templatePath);
        
        return $response->write($template);
    }
}